#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <pcap.h>
#include <err.h>
#include <string.h>
#include <time.h>
#include <getopt.h>

#define DEF_FMT "%s"
#define START_OPT  1001
#define STOP_OPT   1002

static const char *kick_cmd = NULL;
static const char *ProgramName = "tcpdump-split";

/* Prototypes */
static void usage(const char *) __attribute__((noreturn));
static void help(void);
#ifdef __linux__
extern char *strptime(const char *s, const char *format, struct tm *tm);
extern int asprintf(char **strp, const char *fmt, ...);
#endif

int
main(int argc, char *argv[])
{
    pcap_t *in = NULL;
    pcap_dumper_t *out = NULL;
    char errbuf[PCAP_ERRBUF_SIZE + 1];
    struct pcap_pkthdr hdr;
    time_t this_bin;
    time_t last_bin = -1;
    time_t modulus = 300;
    const u_char *data;
    int ch;
    char *p;
    unsigned long ul;
    char *fmt = NULL;
    /* Track of starting and ending time */
    static time_t start_time = 0;
    static time_t stop_time = 0;
    char fname[128];
    char pname[140];

    while ((ch = getopt(argc, argv, "?hB:E:t:f:k:")) != -1)
    {
        switch (ch) {
            case 't':
                /* Validate that is a integer */
                ul = strtoul(optarg, &p, 0);
                if (*p != '\0')
                    usage("argument to -t must be an integer");
                modulus = (unsigned) ul;
                break;
            case 'f':
                fmt = strdup(optarg);
                break;
            case 'k':
                kick_cmd = strdup(optarg);
                break;
            case '?':
            case 'h':
                help();
                exit(0);
                break;
            case 'B':
                {
                struct tm tm;
                memset(&tm, '\0', sizeof(tm));
                if (NULL == strptime(optarg, "%Y-%m-%d %T", &tm))
                    usage("-B arg must have format YYYY-MM-DD HH:MM:SS");
                start_time = timegm(&tm);
                }
                break;
            case 'E':
                {
                struct tm tm;
                memset(&tm, '\0', sizeof(tm));
                if (NULL == strptime(optarg, "%Y-%m-%d %T", &tm))
                    usage("-E arg must have format YYYY-MM-DD HH:MM:SS");
                stop_time = timegm(&tm);
                }
                break;
            default:
                usage("unrecognized command line option");
        }
    }
    if (start_time && stop_time && start_time > stop_time)
        usage("start time must be before stop time");

    /* If not format given, use the default */
    if (NULL == fmt)
        fmt = strdup(DEF_FMT);

    in = pcap_open_offline("-", errbuf);
    if (NULL == in) {
        fprintf(stderr, "stdin: %s", errbuf);
        exit(1);
    }
    while ((data = pcap_next(in, &hdr))) {
        this_bin = hdr.ts.tv_sec - (hdr.ts.tv_sec % modulus);
        /* Check if the packet is within the time window we are
         * interested
         */
        if (start_time && hdr.ts.tv_sec < start_time)
            continue;
        if (stop_time && hdr.ts.tv_sec >= stop_time)
            break;
        if (this_bin != last_bin) {
            if (out) {
                char *cmd = NULL;
                pcap_dump_close(out);
		if (rename(pname, fname) < 0) {
			perror(fname);
			exit(1);
		}
                if (kick_cmd != NULL) {
                    if (asprintf(&cmd, "%s %s &", kick_cmd, fname) < 0){
                        perror("asprintf");
                        cmd = NULL;
                    }
                    else {
                        system(cmd);
                        free(cmd);
                    }
                }
            }
            strftime(fname, 128, fmt, gmtime(&this_bin));
	    snprintf(pname, 140, "%s.part", fname);
            out = pcap_dump_open(in, pname);
            if (NULL == out) {
                perror(pname);
                exit(1);
            }
            last_bin = this_bin;
        }
        pcap_dump((void *)out, &hdr, data);
    }
    if (out) {
        char *cmd = NULL;
        pcap_dump_close(out);
	if (rename(pname, fname) < 0) {
		perror(fname);
		exit(1);
	}
        if (kick_cmd != NULL) {
            if (asprintf(&cmd, "%s %s &", kick_cmd, fname) < 0){
                perror("asprintf");
                cmd = NULL;
            }
            else {
                system(cmd);
                free(cmd);
            }
        }
    }
    exit(0);
}

static void
usage(const char *msg) {
    fprintf(stderr, "%s: usage error: %s\n", ProgramName, msg);
    fprintf(stderr, "\n");
    exit(1);
}

static void
help(void) {
    fprintf(stderr, "%s\n", ProgramName);
    fprintf(stderr,
        "\noptions:\n"
        "\t-? or -\?  print these instructions and exit\n"
        "\t-B YYYY-MM-DD HH:MM:SS        select packets starting on that date\n"
        "\t-E YYYY-MM-DD HH:MM:SS        select packets until this time/date\n"
        "\t-t <interval>                 each <interval> seconds since\n"
        "\t                              the start time indicated by '-B'\n"
        "\t                              will close the old destination file\n"
        "\t                              and create a new one.  Default 300.\n"
        "\t-f <format>                   receives a format accepted by\n"
        "\t                              strftime to name the files created.\n"
        "\t                              Default %%s (UNIX timestamp)\n"
        "\t-k <kick command>             After closing an old destination\n"
        "\t                              file, will execute this command with\n"
        "\t                              the file name as first parameter\n"
    );
}
