/*
 * Copyright (c) 2016-2017, OARC, Inc.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. Neither the name of the copyright holder nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "hashtbl.h"

#include <unistd.h>
#include <stdlib.h>
#include <assert.h>

hashtbl* hash_create(int N, hashfunc* hasher, hashkeycmp* cmp, hashfree* datafree)
{
    hashtbl* new = calloc(1, sizeof(*new));
    assert(new);
    new->modulus  = N;
    new->hasher   = hasher;
    new->keycmp   = cmp;
    new->datafree = datafree;
    new->items    = calloc(N, sizeof(hashitem*));
    return new;
}

int hash_add(const void* key, void* data, hashtbl* tbl)
{
    hashitem* new = calloc(1, sizeof(*new));
    hashitem** I;
    int        slot;
    new->key  = key;
    new->data = data;
    slot      = tbl->hasher(key) % tbl->modulus;
    for (I = &tbl->items[slot]; *I; I = &(*I)->next)
        ;
    *I = new;
    return 0;
}

void* hash_find(const void* key, hashtbl* tbl)
{
    int       slot = tbl->hasher(key) % tbl->modulus;
    hashitem* i;
    for (i = tbl->items[slot]; i; i = i->next) {
        if (0 == tbl->keycmp(key, i->key))
            return i->data;
    }
    return NULL;
}

int hash_count(hashtbl* tbl)
{
    int slot;
    int count = 0;
    for (slot = 0; slot < tbl->modulus; slot++) {
        hashitem* i;
        for (i = tbl->items[slot]; i; i = i->next)
            count++;
    }
    return count;
}

void hash_remove(const void* key, hashtbl* tbl)
{
    hashitem **I, *i;
    int        slot;
    slot = tbl->hasher(key) % tbl->modulus;
    for (I = &tbl->items[slot]; *I; I = &(*I)->next) {
        if (0 == tbl->keycmp(key, (*I)->key)) {
            i  = *I;
            *I = (*I)->next;
            if (i->data)
                tbl->datafree(i->data);
            free(i);
            break;
        }
    }
}

void hash_free(hashtbl* tbl)
{
    int slot;
    for (slot = 0; slot < tbl->modulus; slot++) {
        hashitem* i;
        hashitem* next;
        for (i = tbl->items[slot]; i; i = next) {
            next = i->next;
            if (tbl->datafree)
                tbl->datafree(i->data);
            free(i);
        }
        tbl->items[slot] = NULL;
    }
}

void hash_destroy(hashtbl* tbl)
{
    hash_free(tbl);
    free(tbl->items);
    free(tbl);
}

static void
hash_iter_next_slot(hashtbl* tbl)
{
    while (tbl->iter.next == NULL) {
        tbl->iter.slot++;
        if (tbl->iter.slot == tbl->modulus)
            break;
        tbl->iter.next = tbl->items[tbl->iter.slot];
    }
}

void hash_iter_init(hashtbl* tbl)
{
    tbl->iter.slot = 0;
    tbl->iter.next = tbl->items[tbl->iter.slot];
    if (NULL == tbl->iter.next)
        hash_iter_next_slot(tbl);
}

void* hash_iterate(hashtbl* tbl)
{
    hashitem* this = tbl->iter.next;
    if (this) {
        tbl->iter.next = this->next;
        if (NULL == tbl->iter.next)
            hash_iter_next_slot(tbl);
    }
    return this ? this->data : NULL;
}
